<?php
/**
 * @package        Akaunting
 * @copyright      2018 Akaunting Inc, akaunting.com
 * @license        GNU/GPL http://www.gnu.org/copyleft/gpl.html
 */

require_once('connector.php');

class AkauntingHelper
{
    public $url;
    public $company;
    public $company_id;
    public $connector;

    public function __construct($url, $company_id, $email, $password)
    {
        $this->url = rtrim($url, '/');
        $this->company_id = $company_id;

        $this->connector = new AkauntingConnector($email, $password);
    }

    public function check()
    {
        $url = $this->url . '/api/arastta/check/' . $this->company_id;

        $response = json_decode($this->connector->get($url));

        if (empty($response)) {
            return false;
        }

        if (isset($response->status_code)) {
            switch ($response->status_code) {
                case '401':
                    return 'auth';
                    break;
                case '404':
                    return 'install';
                    break;
                default;
                    return 'connect';
                    break;
            }
        }

        $this->company = $response->data;

        return true;
    }

    public function storeProduct($data)
    {
        $data['company_id'] = $this->company_id;

        // Check if exists
        $product = $this->getResource($data['sku'], 'products');
        if (is_object($product)) {
            return $this->updateProduct($data);
        }

        $url = $this->url . '/api/arastta/products';

        return $this->connector->post($url, $data);
    }

    public function updateProduct($data)
    {
        $data['company_id'] = $this->company_id;

        // Get item object
        $item = $this->getResource($data['sku'], 'products');

        if (!is_object($item)) {
            return $this->storeProduct($data);
        }

        $url = $this->url . '/api/arastta/products/' . $item->id;

        return $this->connector->patch($url, $data);
    }

    public function storeCustomer($data)
    {
        $data['company_id'] = $this->company_id;
        $data['currency_code'] = $this->company->default_currency;

        // Check if exists
        $customer = $this->getResource($data['email'], 'customers');
        if (is_object($customer)) {
            return $this->updateCustomer($data);
        }

        $url = $this->url . '/api/arastta/customers';

        return $this->connector->post($url, $data);
    }

    public function updateCustomer($data)
    {
        $data['company_id'] = $this->company_id;

        // Get customer object
        $customer = $this->getResource($data['email'], 'customers');

        if (!is_object($customer)) {
            return $this->storeCustomer($data);
        }

        $data['currency_code'] = $this->company->default_currency;

        $url = $this->url . '/api/arastta/customers/' . $customer->id;

        return $this->connector->patch($url, $data);
    }

    public function storeOrder($data)
    {
        $data['company_id'] = $this->company_id;

        // Get company object
        if (!empty($data['customer_id'])) {
            $customer = $this->getResource($data['email'], 'customers');
        } else {
            $this->storeCustomer([
                'name' => $data['firstname'] . ' ' . $data['lastname'],
                'email' => $data['email'],
                'phone' => $data['telephone'],
                'address' => '',
                'enabled' => 1,
            ]);

            $customer = $this->getResource($data['email'], 'customers');
        }

        $data['customer_id'] = $customer->id;

        $sub_total = 0;

        $data['item'] = [];

        foreach ($row['products'] as $product) {
            $data['item'][] = $product;

            $sub_total += $product['price'] * $product['quantity'];
        }

        $data['totals'] = [];

        $sort_order = 1;

        // add order total
        foreach ($data['total'] as $order_total) {
            $t = [
                'code'                  =>  $order_total['code'],
                'name'                  =>  $order_total['title'],
                'amount'                =>  $order_total['value'],
                'sort_order'            =>  $sort_order,
                'company_id'            =>  $this->company_id,
            ];

            $data['totals'][] = $t;

            $sort_order++;
        }

        $url = $this->url . '/api/arastta/orders';

        $headers = $this->connector->post($url, $data);

        if (empty($headers)) {
            return;
        }

        // Lets add payment
        $data = AkauntingTransformer::order_payment($row);

        $data['company_id'] = $this->company_id;
        $data['account_id'] = $this->company->default_account;
        $data['payment_method'] = $this->company->default_payment_method;

        $invoice_id = end(explode('/', $headers['Location']));

        $url .= '/' . $invoice_id . '/payments';

        // Check if exists
        $order = $this->getResource($data['invoice_number'], 'orders');
        if (is_object($order)) {
            return $this->updateOrder($data);
        }

        $url = $this->url . '/api/arastta/orders';

        $headers = $this->connector->post($url, $data);

        if (empty($headers)) {
            return false;
        }

        return $headers;
    }

    public function updateOrder($data)
    {
        $data['company_id'] = $this->company_id;

        // Get invoice object
        $invoice = $this->getResource($data['invoice_number'], 'orders');

        if (!is_object($invoice)) {
            return false;
        }

        $url = $this->url . '/api/arastta/orders/' . $invoice->id;

        return $this->connector->patch($url, $data);
    }

    public function getResource($id, $type)
    {
        static $list = array(
            'products' => array(),
            'customers' => array(),
            'orders' => array(),
        );

        if (!isset($list[$type][$id])) {
            $url = $this->url . '/api/arastta/'. $type .'/' . $id;

            $url .= '?company_id=' . $this->company_id;

            $response = $this->connector->get($url);

            if (empty($response)) {
                $list[$type][$id] = null;
            } else {
                $obj = json_decode($response);

                if (is_object($obj) && isset($obj->data)) {
                    $list[$type][$id] = $obj->data;
                } else {
                    $list[$type][$id] = null;
                }
            }
        }

        return $list[$type][$id];
    }
}